/// @file syslogLogger.cpp
///
/// @brief Implements the syslog logger
///
/// @component Uspi/DeviceDetector
///
/// @author F.Berat / ADITG/SWG / fberat@de.adit-jv.com
///
/// @copyright (c) 2016 Advanced Driver Information Technology.
/// This code is developed by Advanced Driver Information Technology.
/// Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
/// All rights reserved.
///
/// @see Log DltLogger

#include <cstring>
#include <iostream>
#include <string>
#include <syslog.h>

#include "syslogLogger.h"
#include "logger.h"

namespace adit {
namespace uspi {

/// @var ddlog
/// @brief The global output stream to be used for logging.
/// Could be replace by clog if we could ensure the library user to properly
/// set the rdbuf.
std::ostream ddlog(new SysLog());

/// @brief SysLog constructor
/// @param ident The identity to use for logging
///
/// The identity is appended to the logs if set. This to avoid confusion with
/// the identity potentially used while executing openlog.
SysLog::SysLog(const char *ident)
{
    mPriority = kLogInfo;
    mDefaultPriority = kLogInfo;

    if (ident && (ident[0] != '\0')) {
        mIdent.assign(ident);
    }
}

/// @brief Flushes the @ref mBuffer to syslog if non-empty
///
/// The buffer is provided to syslog using @ref mPriority for the log level. If
/// @ref mIdent is not empty, it is appended at the beginning of the log, in the
/// form @c "mIdent: mBuffer". Once the buffer is sent, it is erased, and
/// the @ref mPriority is reset to the @ref mDefaultPriority value.
int SysLog::sync()
{
    if (mBuffer.length()) {
        if (mIdent != "")
            syslog(mPriority, "%s: %s", mIdent.c_str(),  mBuffer.c_str());
        else
            syslog(mPriority, "%s",  mBuffer.c_str());

        mBuffer.erase();
        mPriority = mDefaultPriority;
    }

    return 0;
}

/// @brief Add a character to the buffer and sync if EOF is met.
/// @param c The character to push into the @ref mBuffer
int SysLog::overflow(int c)
{
    if (c != EOF)
        mBuffer += static_cast<char>(c);
    else
        sync();

    return c;
}

} // namespace uspi
} // namespace adit
